using System;
using System.Collections;

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>2/27/2005</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///	List of business objects implementing <see cref="IRecognizableByID"/> interface,
	///	sorted by some property (defined by supplied comparer) and hashed by object-specific ID.
	///	</summary>

	#endregion

	public class BusinessObjectSortedHashList : ICollection	
	{
		private const int DefaultInitialCapacity = 10;

		private readonly ArrayList _list;
		private readonly Hashtable _hashtable;
		private readonly IComparer _comparer;
		private bool _isCurrentlySorted;

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6713"> 
		///		<ExpectedInput>Non-null comparer.</ExpectedInput>
		///		<ExpectedOutput>An instance of the class.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6714"> 
		///		<ExpectedInput>Null instead of comparer.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Simple constructor creating empty list. 
		/// </summary>
		/// <param name="comparer">Comparer to use for sorting business objects in the list.</param>
		public BusinessObjectSortedHashList( IComparer comparer ) : 
			this( comparer, DefaultInitialCapacity ) {}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6715"> 
		///		<ExpectedInput>Non-null comparer, valid initial list capactity.</ExpectedInput>
		///		<ExpectedOutput>An instance of the class.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6716"> 
		///		<ExpectedInput>Null comparer, valid initial capacity.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6717"> 
		///		<ExpectedInput>Valid comparer, negative integer instead of initial capacity.</ExpectedInput>
		///		<ExpectedOutput>ArgumentOutOfRangeException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor creating empty list with specified initial capacity.
		/// </summary>
		/// <param name="comparer">Comparer to use for sorting business objects in the list.</param>
		/// <param name="initialCapacity">Initial capacity of the list.</param>
		public BusinessObjectSortedHashList( IComparer comparer, int initialCapacity )
		{
			if( comparer == null )
				throw( new ArgumentNullException( "comparer" ) );

			_list = new ArrayList( initialCapacity );
			_hashtable = new Hashtable( initialCapacity );
			_comparer = comparer;

			_isCurrentlySorted = true;
		}

		/// <summary>
		/// Ensures that the list is sorted by invoking sort if needed. 
		/// </summary>
		protected virtual void EnsureListIsSorted()
		{
			if( _isCurrentlySorted )
				return;

			_list.Sort( _comparer );
			_isCurrentlySorted = true;
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6718"> 
		///		<ExpectedInput>Valid business object to add.</ExpectedInput>
		///		<ExpectedOutput>Reference to supplied object, count incremented by 1.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6719"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Adds supplied business object to the list.
		/// </summary>
		/// <param name="objectToAdd">Non-null object to add.</param>
		/// <returns>Reference to the added object.</returns>
		public IRecognizableByID Add( IRecognizableByID objectToAdd )
		{
			if( objectToAdd == null )
				throw( new ArgumentNullException( "objectToAdd" ) );

			_list.Add( objectToAdd );
			_hashtable.Add( objectToAdd.GetObjectID(), objectToAdd );

			_isCurrentlySorted = false;

			return objectToAdd;
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6720"> 
		///		<ExpectedInput>Valid object ID for an object in the list.</ExpectedInput>
		///		<ExpectedOutput>Reference to the object with provided ID.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6722"> 
		///		<ExpectedInput>Valid object ID for an object that is not in the list.</ExpectedInput>
		///		<ExpectedOutput>Null.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6721"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves an object by its ID.
		/// </summary>
		/// <param name="objectID">Object ID (key in the list).</param>
		/// <returns>Business object in the list or null if the object is not found.</returns>
		public IRecognizableByID GetByObjectID( object objectID )
		{			
			return (IRecognizableByID)_hashtable[ objectID ];
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6723"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>Non-null enumerator set at the list.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6724"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets enumerator for the list.
		/// </summary>
		/// <returns>List enumberator.</returns>
		public IEnumerator GetEnumerator()
		{			
			EnsureListIsSorted();

			return _list.GetEnumerator();
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6725"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>Count of the objects in the list.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6726"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the number of elements contained in the list.
		/// </summary>
		public int Count
		{
			get
			{
				return _list.Count;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6727"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>Non-null object.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6728"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets an object that can be used to synchronize access to the list.
		/// </summary>
		public object SyncRoot
		{
			get
			{
				return this;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6729"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>False.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6730"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets a value indicating whether access to the list is synchronized (thread-safe) which 
		/// is always false - the list is never thread safe and caller must take care of thread safe access.
		/// </summary>
		public bool IsSynchronized
		{
			get
			{
				return false;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6731"> 
		///		<ExpectedInput>1-element list and 1-element array.</ExpectedInput>
		///		<ExpectedOutput>The element from the list is copied to the array.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6733"> 
		///		<ExpectedInput>Empty list and 1-element array.</ExpectedInput>
		///		<ExpectedOutput>Nothing is copied to the array, it remais intact.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6732"> 
		///		<ExpectedInput>Null instead of array.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6734"> 
		///		<ExpectedInput>Negative copy-to start index.</ExpectedInput>
		///		<ExpectedOutput>ArgumentOutOfRangeException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Copies the elements of the list to an <see cref="Array"/>, starting at a particular <see cref="Array"/> index.
		/// </summary>
		/// <param name="array"><see cref="Array"/> to copy elements to.</param>
		/// <param name="index">Starting index in the <see cref="Array"/>.</param>
		public void CopyTo( Array array, int index )
		{
			_list.CopyTo( array, index );
		}
	}
}
